const log4js = require("log4js");
const moment = require("moment");
const fs = require("fs");
const path = require("path");

// 获取日志文件路径
function returnLogFilePath() {
  const today = moment(new Date()).format("YYYY-MM-DD");
  const low_path = process.cwd();
  let log_dir = path.join(low_path, "logs");

  if (!fs.existsSync(log_dir)) {
    fs.mkdirSync(log_dir, { recursive: true });
  }

  const dir = path.join(low_path, `/logs/${today}`);
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir);
  }

  return path.join(dir, "api.log");
}

// 获取错误日志文件路径
function returnErrorLogFilePath() {
  const today = moment(new Date()).format("YYYY-MM-DD");
  const low_path = process.cwd();
  let log_dir = path.join(low_path, "logs");

  if (!fs.existsSync(log_dir)) {
    fs.mkdirSync(log_dir, { recursive: true });
  }

  const dir = path.join(low_path, `/logs/${today}`);
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir);
  }

  return path.join(dir, "error.log");
}

// 初始化日志系统
function initLogger() {
  const logFilePath = returnLogFilePath();
  const errorLogFilePath = returnErrorLogFilePath();

  log4js.configure({
    appenders: {
      console: { type: "console" },
      file: {
        type: "file",
        filename: logFilePath,
        maxLogSize: 1024 * 1000 * 10,
        backups: 5,
        compress: true,
      },
      errorFile: {
        type: "file",
        filename: errorLogFilePath,
        maxLogSize: 1024 * 1000 * 5,
        backups: 10,
        compress: true,
      },
      errors: {
        type: "logLevelFilter",
        appender: "errorFile",
        level: "error",
      },
    },
    categories: {
      default: {
        appenders: ["console", "file", "errors"],
        level: "debug",
      },
    },
  });

  return log4js.getLogger();
}

// 创建全局日志实例
const logger = initLogger();

// 添加请求日志
const addFormatLog = function(req, res, data) {
  // 检查是否需要屏蔽此URL
  if (shouldSkipLogging(req.url)) {
    return;
  }

  const now = new Date();
  const resTime = now - req._startTime;

  const token = req.headers["jwt-token"];
  const {
    ip,
    headers,
    method,
    url,
    body,
    httpVersion,
    res: { statusCode, _headers },
  } = req;

  let logInfo = {
    ip,
    host: headers.host,
    resTime,
    method,
    url,
    token,
    body,
    httpVersion,
    statusCode,
    contentLength: _headers["content-length"],
    userAgent: headers["user-agent"],
    data: data[0],
  };

  logger.info(` 
    时间：${moment(new Date()).format("YYYY-MM-DD HH:mm:ss")}
    ip : ${logInfo.ip}
    host : ${logInfo.host}
    响应时间 : ${logInfo.resTime / 1000} s 
    method ：${logInfo.method}
    url ：${logInfo.url} 
    token ：${logInfo.token || ""}
    body ：${JSON.stringify(logInfo.body)}
    状态码 ：${logInfo.statusCode}
    data ：${JSON.stringify(logInfo.data)} 
   `);
};

// 判断是否需要跳过日志记录
function shouldSkipLogging(url) {
  // 定义需要屏蔽的URL列表
  const skippedUrls = [
    "/getAllCountry", // 您想要屏蔽的接口
    // 可以添加更多需要屏蔽的URL
  ];

  return skippedUrls.some((skippedUrl) => url.includes(skippedUrl));
}

// 日志中间件
export const logMiddleWare = () => {
  return function(req, res, next) {
    // 检查是否需要跳过此请求的日志记录
    if (shouldSkipLogging(req.url)) {
      return next();
    }

    req._startTime = new Date();

    const oldSend = res.send;
    res.send = function() {
      oldSend.apply(res, arguments);
      if (typeof [...arguments][0] === "object") {
        res.once("finish", () => addFormatLog(req, res, arguments));
      }
    };

    return next();
  };
};

// ================= 错误捕获增强 =================

// 捕获未处理的异常
process.on("uncaughtException", (error) => {
  logger.fatal(`
    ====== 未捕获的异常 ======
    时间: ${moment().format("YYYY-MM-DD HH:mm:ss")}
    错误信息: ${error.message}
    堆栈跟踪:
    ${error.stack}
    ========================
  `);
});

// 捕获未处理的Promise拒绝
process.on("unhandledRejection", (reason, promise) => {
  logger.fatal(`
    ====== 未处理的Promise拒绝 ======
    时间: ${moment().format("YYYY-MM-DD HH:mm:ss")}
    拒绝原因: ${reason}
    Promise对象: ${promise}
    ===============================
  `);
});

// 导出增强的日志系统
export default {
  logger,
  logMiddleWare,
  // 添加错误日志函数
  logError: (error, context = {}) => {
    logger.error(`
      ====== 错误日志 ======
      时间: ${moment().format("YYYY-MM-DD HH:mm:ss")}
      错误信息: ${error.message}
      堆栈跟踪:
      ${error.stack}
      上下文信息:
      ${JSON.stringify(context, null, 2)}
      =====================
    `);
  },
  // 添加致命错误日志函数
  logFatal: (error, context = {}) => {
    logger.fatal(`
      ====== 致命错误 ======
      时间: ${moment().format("YYYY-MM-DD HH:mm:ss")}
      错误信息: ${error.message}
      堆栈跟踪:
      ${error.stack}
      上下文信息:
      ${JSON.stringify(context, null, 2)}
      =====================
    `);
  },
};
