<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Joshine\Catalog\Model\Product\Gallery;

use Magento\Catalog\Api\Data\ProductAttributeMediaGalleryEntryInterface;
use Magento\Catalog\Api\Data\ProductInterfaceFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\ImageContent;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\StateException;
use Magento\Framework\Api\ImageContentValidatorInterface;
use Magento\Framework\View\Asset\NotationResolver\Variable;
use Magento\Framework\Webapi\Rest\Request;

/**
 * Class GalleryManagement
 *
 * Provides implementation of api interface ProductAttributeMediaGalleryManagementInterface
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class GalleryManagement implements \Joshine\Catalog\Api\ProductAttributeMediaGalleryManagementInterface
{
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    protected $productRepository;

    /**
     * @var ImageContentValidatorInterface
     */
    protected $contentValidator;

    /**
     * @var ProductInterfaceFactory
     */
    private $productInterfaceFactory;


    protected  $entry;


    /**
     * @param ProductRepositoryInterface $productRepository
     * @param ImageContentValidatorInterface $contentValidator
     * @param ProductInterfaceFactory|null $productInterfaceFactory
     * @param DeleteValidator|null $deleteValidator
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        ProductRepositoryInterface $productRepository,
        ImageContentValidatorInterface $contentValidator,
        ProductAttributeMediaGalleryEntryInterface $entry,
        ?ProductInterfaceFactory $productInterfaceFactory = null

    ) {
        $this->productRepository = $productRepository;
        $this->contentValidator = $contentValidator;
        $this->productInterfaceFactory = $productInterfaceFactory
            ?? ObjectManager::getInstance()->get(ProductInterfaceFactory::class);
        $this->entry = $entry;
    }


    /**
     * @inheritdoc
     */
    public function add($content)
    {
        if (empty($content)) {
            throw new StateException(__("request   update date is null ,so failed"));
        }
        $ids = [];
        $sku = $content[0]['sku'];
        foreach ($content as  $entry){
            $entry_id = 0;
            if (key_exists("id",$entry)){
                $entry_id= (int)$entry["id"];
            }
            $this->entry->setMediaType($entry['media_type']);
            $this->entry->setDisabled($entry['disabled']);
            $this->entry->setPosition($entry['position']);
            $this->entry->setLabel($entry['label']);
            $this->entry->setTypes($entry['types']);

            if ($entry_id){
                $this->entry->setId($entry_id);
                $result = $this->update($sku,$this->entry);
                if ($result){
                    $id = $entry_id;
                }else{
                    throw new StateException(__("$entry_id update failed"));
                }
            }else{
                $this->entry->setFile($entry['file']);
                $imageContent = new ImageContent();
                $imageContent->setBase64EncodedData($entry['content']['base64_encoded_data']);
                $imageContent->setName($entry['content']['name']);
                $imageContent->setType($entry['content']['type']);
                $this->entry->setContent($imageContent);
                $id  = $this->create($sku,$this->entry);
            }
            if (is_numeric((int)$id)){
                $ids []= (int)$id;
            }
            $this->entry->setData([]);
        }
        if (!$ids){
            throw new StateException(__('Batch upload of product images failed'));
        }

        $list =  $this->getList($sku);
        foreach ($list as $key => $v) {
            if (!in_array($v->getId(),$ids)) {
                unset($list[$key]);
            }
        }
        array_multisort($list);
        return $list;
    }


    /**
     * @inheritdoc
     */
    public function create($sku,ProductAttributeMediaGalleryEntryInterface $entry)
    {
        /** @var $entry ProductAttributeMediaGalleryEntryInterface */
        $entryContent = $entry->getContent();
        if (!$this->contentValidator->isValid($entryContent)) {
            throw new InputException(__('The image content is invalid. Verify the content and try again.'));
        }
        $product = $this->productRepository->get($sku, true);

        $existingMediaGalleryEntries = $product->getMediaGalleryEntries();

        $existingEntryIds = [];
        if ($existingMediaGalleryEntries == null) {
            // set all media types if not specified
            if ($entry->getTypes() == null) {
                $entry->setTypes(array_keys($product->getMediaAttributes()));
            }
            $existingMediaGalleryEntries = [$entry];
        } else {
            foreach ($existingMediaGalleryEntries as $existingEntries) {
                $existingEntryIds[$existingEntries->getId()] = $existingEntries->getId();
            }
            $existingMediaGalleryEntries[] = $entry;
        }
        $product = $this->productInterfaceFactory->create();
        $product->setSku($sku);
        $product->setMediaGalleryEntries($existingMediaGalleryEntries);
        try {
            $product = $this->productRepository->save($product);
        } catch (\Exception $e) {
            if ($e instanceof InputException) {
                throw $e;
            } else {
                throw new StateException(__("The product can't be saved."));
            }
        }

        foreach ($product->getMediaGalleryEntries() as $entry) {
            if (!isset($existingEntryIds[$entry->getId()])) {
                return $entry->getId();
            }
        }
        throw new StateException(__('The new media gallery entry failed to save.'));
    }


    /**
     * @inheritdoc
     */
    public function update($sku, ProductAttributeMediaGalleryEntryInterface $entry)
    {
        $product = $this->productRepository->get($sku, true);
        $existingMediaGalleryEntries = $product->getMediaGalleryEntries();
        if ($existingMediaGalleryEntries == null) {
            throw new NoSuchEntityException(
                __('No image with the provided ID was found. Verify the ID and try again.')
            );
        }
        $found = false;
        $entryTypes = (array)$entry->getTypes();
        foreach ($existingMediaGalleryEntries as $key => $existingEntry) {
            $existingEntryTypes = (array)$existingEntry->getTypes();
            $existingEntry->setTypes(array_diff($existingEntryTypes, $entryTypes));

            if ($existingEntry->getId() == $entry->getId()) {
                $found = true;
                $existingMediaGalleryEntries[$key] = $entry;
            }
        }
        if (!$found) {
            throw new NoSuchEntityException(
                __('No image with the provided ID was found. Verify the ID and try again.')
            );
        }
        $product = $this->productInterfaceFactory->create();
        $product->setSku($sku);
        $product->setMediaGalleryEntries($existingMediaGalleryEntries);

        try {
            $this->productRepository->save($product);
        } catch (\Exception $exception) {
            throw new StateException(__("The product can't be saved."));
        }
        return true;
    }



    /**
     * @inheritdoc
     */
    public function getList($sku)
    {
        /** @var \Magento\Catalog\Model\Product $product */
        $product = $this->productRepository->get($sku);

        return $product->getMediaGalleryEntries();
    }

}
