<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Joshine\Category\Block\Html;

use Magento\Framework\Data\Tree\Node;
use Magento\Framework\Data\Tree\Node\Collection;
use Magento\Theme\Block\Html\Topmenu as JoshineMenu;

/**
 * Html page top menu block
 *
 * @api
 * @since 100.0.2
 */
class Topmenu extends JoshineMenu
{
    /**
     * Remove children from collection when the parent is not active
     *
     * @param Collection $children
     * @param int $childLevel
     * @return void
     */
    private function removeChildrenWithoutActiveParent(Collection $children, int $childLevel): void
    {

        /** @var Node $child */
        foreach ($children as $child) {
            if ($childLevel === 0 && $child->getData('is_parent_active') === false) {
                $children->delete($child);
            }
        }
    }

    /**
     * Retrieve child level based on parent level
     *
     * @param int $parentLevel
     *
     * @return int
     */
    private function getChildLevel($parentLevel): int
    {
        return $parentLevel === null ? 0 : $parentLevel + 1;
    }

    /**
     * Check if new column should be added.
     *
     * @param array $colBrakes
     * @param int $counter
     * @return bool
     */
    private function shouldAddNewColumn(array $colBrakes, int $counter): bool
    {
        return count($colBrakes) && $colBrakes[$counter]['colbrake'];
    }

    protected function _getMenuItemClasses(\Magento\Framework\Data\Tree\Node $item)
    {
        $classes = [
            'level' . $item->getLevel(),
            $item->getPositionClass(),
        ];

        if ($item->getIsCategory()) {
            $classes[] = 'category-item';
        }

        if ($item->getLevel() == 0 ) {
            $classes[] = 'level-top';
        }

        if ($item->getIsFirst()) {
            $classes[] = 'first';
        }

        if ($item->getIsActive()) {
            $classes[] = 'active';
        } elseif ($item->getHasActive()) {
            $classes[] = 'has-active';
        }

        if ($item->getIsLast()) {
            $classes[] = 'last';
        }

        if ($item->getClass()) {
            $classes[] = $item->getClass();
        }

        if ($item->hasChildren()) {
            $classes[] = 'parent';
        }

        if ($item->getLevel() != 0) {
            $classes = $this->_getColumns($classes, $item);
        }

        return $classes;
    }

    private function _getColumns($classes, $item)
    {
        $parentSetMyColumns = $item->getParent()->getSubmenuColumnsSingle();
        $selfColumns    = $item->getSelfColumns();
        $defaultColumns = '3';
        if ($item->getLevel() > 1) {
            $defaultColumns = '12';
        }
        $columns = ($selfColumns ?? $parentSetMyColumns) ?? $defaultColumns;
        $classes[] = 'joshine-col-lg-'.$columns;
        return $classes;
    }

    protected function _getHtml(
        Node $menuTree,
             $childrenWrapClass,
             $limit,
        array $colBrakes = []
    ) {

        $html = '';

        $children = $menuTree->getChildren();
        $childLevel = $this->getChildLevel($menuTree->getLevel());

        $this->removeChildrenWithoutActiveParent($children, $childLevel);

        $counter = 1;
        $childrenCount = $children->count();

        $parentPositionClass = $menuTree->getPositionClass();
        $itemPositionClassPrefix = $parentPositionClass ? $parentPositionClass . '-' : 'nav-';

        /** @var Node $child */
        foreach ($children as $child) {
            $child->setLevel($childLevel);
            $child->setIsFirst($counter === 1);
            $child->setIsLast($counter === $childrenCount);
            $child->setPositionClass($itemPositionClassPrefix . $counter);

            $outermostClassCode = '';
            $outermostClass = $menuTree->getOutermostClass();

            if ($childLevel === 0 && $outermostClass) {
                $outermostClassCode = ' class="' . $outermostClass . '" ';
                $this->setCurrentClass($child, $outermostClass);
            }

            if ($this->shouldAddNewColumn($colBrakes, $counter)) {
                $html .= '</ul></li><li class="column"><ul>';
            }

            $html .= '<li ' . $this->_getRenderedMenuItemAttributes($child) . '>';
            $html .= '<a href="' . $child->getUrl() . '" ' . $outermostClassCode . '><span>' . $this->escapeHtml(
                    $child->getName()
                ) . '</span>'.$this->_addBadge($child).'</a>' . $this->_addSubMenu(
                    $child,
                    $childLevel,
                    $childrenWrapClass,
                    $limit
                ) . '</li>';
            $counter++;
        }

        if (is_array($colBrakes) && !empty($colBrakes) && $limit) {
            $html = '<li class="column"><ul>' . $html . '</ul></li>';
        }

        return $html;
    }


    private function _addBadge($child )
    {
        $badge = $child->getBadge();
        if (!$badge || !json_decode($badge)) {
            return "";
        }
        $json  = json_decode($badge, true);
        $class = $json['class'];
        $text  = $json['text'];
        return "<span class='{$class} menu-badge joshine-hidden'>".$this->escapeHtml($text) .'</span>';
    }

    protected function _addSubMenu($child, $childLevel, $childrenWrapClass, $limit)
    {
        $html = '';
        if (!$child->hasChildren()) {
            return $html;
        }

        $colStops = [];
        if ($childLevel == 0 && $limit) {
            $colStops = $this->_columnBrake($child->getChildren(), $limit);
        }

        $sumMenuColumnsTotal = $child->getSubmenuColumnsTotal();
        $columns = $sumMenuColumnsTotal ?? "11" ;
        if (intval($columns) >= 12) {
            $columns = "11";
        }
        if ($childLevel == 0) {
            $html  = "<div class=\"level{$childLevel} {$childrenWrapClass}\">";
            $html .= "<div class=\"row  \">
                    <ul class=\"subchildmenu joshine-col-lg-push-1 mega-columns joshine-clearfix joshine-col-lg-{$columns} {$child->getMenuImage()}\">
                        {$this->_getHtml($child, $childrenWrapClass, $limit, $colStops)}
                    </ul>";
            $html .= $this->appendImage($child);
            $html .= "</div></div>";
        } else {
            $html .= '<ul class="level' . $childLevel . '">';
            $html .= $this->_getHtml($child, $childrenWrapClass, $limit, $colStops);
            $html .= '</ul>';
        }
        return $html;
    }

    private function appendImage($child)
    {
        $html = "";
        if (!$child->getMenuImage()) {
            return $html;
        }

        if ($child->getMenuImage()) {
            $html  .= "<div class=\" pull-right joshine-col-lg-3 joshine-col-lg-push-1 joshine-hidden-xs joshine-hidden-sm joshine-hidden-md  \">";
            $html  .=  "<img src='{$child->getMenuImage()}'/>";
            $html  .= "</div>";
        }

        return $html;
    }

}
