<?php
namespace Joshine\Review\Controller\Ajax;

use Amasty\AdvancedReview\Model\ResourceModel\Images as ImagesModel;
use Joshine\Review\Helper\BlockHelper;
use Joshine\Review\Helper\ImageHelper;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Review\Model\ResourceModel\Review\Collection;
use Magento\Review\Model\Review;

class ReviewInfo extends \Magento\Framework\App\Action\Action {


    /**
     * @var \Magento\Framework\Data\Form\FormKey\Validator
     */
    private $formKeyValidator;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @var \Magento\Framework\Json\EncoderInterface
     */
    private $jsonEncoder;

    /**
     * @var JsonFactory
     */
    private $resultJsonFactory;


    private $reviewFactory;

    /**
     * @var ProductFactory
     */
    private $productFactory;

    /**
     * @var \Magento\Review\Model\ResourceModel\Review\CollectionFactory
     */
    private $reviewsColFactory;
    /**
     * @var \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress
     */
    private $remoteAddress;

    public function __construct(
        Context $context,
        \Psr\Log\LoggerInterface $logger,
        \Magento\Framework\Json\EncoderInterface $jsonEncoder,
        JsonFactory $resultJsonFactory,
        \Magento\Review\Model\ReviewFactory $reviewFactory,
        ProductFactory $productFactory,
        \Magento\Review\Model\ResourceModel\Review\CollectionFactory  $reviewsColFactory,
        \Magento\Framework\Data\Form\FormKey\Validator $formKeyValidator,
        \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress $remoteAddress
    ) {
        parent::__construct($context);
        $this->logger = $logger;
        $this->jsonEncoder = $jsonEncoder;
        $this->resultJsonFactory  = $resultJsonFactory;
        $this->reviewFactory      = $reviewFactory;
        $this->productFactory     = $productFactory;
        $this->reviewsColFactory  = $reviewsColFactory;
        $this->formKeyValidator   = $formKeyValidator;
        $this->remoteAddress      = $remoteAddress;
    }

    /**
     * @return \Magento\Framework\App\ResponseInterface|\Magento\Framework\Controller\Result\Json|\Magento\Framework\Controller\ResultInterface
     */
    public function execute()
    {
        $message = [
            'error' => __('Sorry. There is a problem with your Request.')
        ];


        if ($this->getRequest()->isAjax()) {
            try {
                if (!$this->formKeyValidator->validate($this->getRequest())) {
                    throw new LocalizedException(
                        __('Form key is not valid. Please try to reload the page.')
                    );
                }
                $reviewId    = $this->getRequest()->getParam('review_id');
                $productId   = $this->getRequest()->getParam('product_id');
                if (empty($reviewId) || empty($productId) ) {
                    throw new LocalizedException(
                        __('request is valid.')
                    );
                }
                $reviewInfo = $this->getReviewInfo($reviewId, $productId);
                $reviews    = $this->getAllReview($productId);
                $message = [
                    'success' => __('Success.'),
                    'data'    => $reviews,
                    'review'  => $reviewInfo,
                    'total'   => count($reviews),
                    'offset'  => $this->getOffset($reviews, $reviewInfo)
                ];

            } catch (LocalizedException $e) {
                $message = ['error' => $e->getMessage()];
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());
            }
        }

        $resultPage = $this->resultJsonFactory->create();
        $resultPage->setHttpResponseCode(200);
        $resultPage->setData($message);
        return $resultPage;
    }

    /**
     * @param $productId
     * @return \Magento\Review\Model\ResourceModel\Review\Collection
     */
    public function getCollection($productId) : Collection
    {
        return $this->reviewsColFactory->create()->addStatusFilter(
            \Magento\Review\Model\Review::STATUS_APPROVED
        )->addEntityFilter(
            'product',
            $productId
        );
    }

    private function getStoreId()
    {
        $storeManager = $this->_objectManager->get('\Magento\Store\Model\StoreManagerInterface');
        return $storeManager->getStore()->getId();
    }

    public function getRatingPercent($reviewId)
    {
        $ratingCollection = $this->_objectManager->create('Magento\Review\Model\ResourceModel\Rating\Option\Vote\Collection')
            ->addFieldToFilter('review_id',$reviewId);
        $data = $ratingCollection->getData();
        return $data[0]['percent'] ?? 100;
    }

    public function getAllReview($productId)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $db = $objectManager->get( 'Magento\Framework\App\ResourceConnection' )->getConnection();
        $tblReview  = $db->getTableName( 'review' );
        $tblImages  = $db->getTableName( 'joshine_review_images' );
        $tblDetail  = $db->getTableName( 'review_detail' );
        $select = $db->select()
            ->distinct()
            ->from( [ 'review' => $tblReview ], ['*'] )
            ->join( [ 'images' => $tblImages ], 'review.review_id = images.review_id' )
            ->join( [ 'review_detail' => $tblDetail ], 'review.review_id = review_detail.review_id' )
            ->where( 'review.entity_pk_value = ?', $productId )
            ->where('review.status_id = ? ', Review::STATUS_APPROVED)
            ->group('review.review_id')->limit(50)->order('review.created_at DESC',);

        $data = $db->fetchAll($select);
        $res = [];
        foreach ($data as $item) {
            $_item['review_id']    = $item['review_id'];
            $_item['size_fits']    = $item['size_fits'];
            $_item['nickname']     = $item['nickname'];
            $_item['title']        = $item['title'];
            $_item['detail']       = $item['detail'];
            $_item['created_at']   = $item['created_at'];
            $_item['images']       = $this->getImages($item['review_id'], $productId);
            $_item['rating']       = $this->getRatingPercent($item['review_id']);
            $_item['helpful']         = $this->getVoteCount($item['review_id'])['plus'];
            $_item['helpful_clicked'] = $this->getVoteCount($item['review_id'], $this->remoteAddress->getRemoteAddress())['plus'] > 0;
            $res[] = $_item;
        }
       return $res;
    }

    public function getReviewInfo($reviewId, $productId)
    {
        $model = $this->reviewFactory->create()->load($reviewId);
        $_item = [];
        $_item['review_id']    = $reviewId;
        $_item['size_fits']    = $model->getSize_fits();
        $_item['nickname']     = $model->getNickname();
        $_item['title']        = $model->getTitle();
        $_item['detail']       = $model->getDetail();
        $_item['created_at']   = $model->getCreatedAt();
        $_item['images']       = $this->getImages($reviewId, $productId);
        $_item['rating']       = $this->getRatingPercent($reviewId);
        $_item['helpful']         = $this->getVoteCount($reviewId)['plus'];
        $_item['helpful_clicked'] = $this->getVoteCount($reviewId, $this->remoteAddress->getRemoteAddress())['plus'] > 0;
        return $_item;
    }

    public function getVoteCount($reviewId, $ip=null)
    {
        $repo = $this->_objectManager->get('Joshine\Review\Model\Repository\VoteRepository');
        return $repo->getVotesCount($reviewId);
    }

    public function getImages($reviewId, $productId)
    {
        $collect = $this->_objectManager->create('Joshine\Review\Model\ResourceModel\Images\Collection')
            ->addFieldToSelect('*')
            ->addFieldToFilter('review_id', $reviewId);

        /** @var  $helper BlockHelper */
        $helper = $this->_objectManager->get('Joshine\Review\Helper\BlockHelper');
        $block  = $helper->getReviewImagesBlock($reviewId, $productId);

        $res = [];
        foreach ($collect->getItems() as $item) {
            $res['full'][]   = ['url'=>$block->getFullImagePath($item), 'image_id'=>$item->getId()] ;
            $res['thumb'][]  = ['url'=>$block->getResizedImagePathByLimit($item, 100), 'image_id'=>$item->getId()] ;   ;
        }
        return $res;
    }

    public function getOffset($allArr, $arr)
    {
        $offset = 0;
        foreach ($allArr as $key => $value) {
            $offset++;
            if ($value['review_id'] == $arr['review_id']) {
                break;
            }
        }
        return $offset;
    }
}