<?php

namespace Joshine\FacebookFeed\Console\Command;

use Magento\Framework\App\State;
use Magento\Store\Model\Store;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Catalog\Model\ProductRepository;
use Magento\Framework\Filesystem;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Catalog\Helper\Image as ImageFactory;
use Magento\CatalogInventory\Model\Stock\StockItemRepository;
use Magento\ConfigurableProduct\Api\LinkManagementInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class SomeCommand
 */
class FacebookFeed extends Command
{
    /* @var ProductRepository */
    private $productRepository;
    /* var SearchCriteriaBuilder */
    private $searchCriteriaBuilder;
    /* var DirectoryList */
    private $directoryList;
    /* var Filesystem */
    private $filesystem;
    /* @var ImageFactory */
    protected $imageHelperFactory;
    /* @var StockItemRepository */
    protected $stockItemRepository;
    /* @var LinkManagementInterface */
    protected $linkManagement;
    /* @var StoreManagerInterface */
    protected $storeManager;

    private $columns = array(
        'id',
        'title',
        'description',
        'availability',
        'condition',
        'price',
        'link',
        'image_link',
        'brand',
        'google_product_category',
        'fb_product_category',
        'quantity_to_sell_on_facebook',
        'sale_price',
        'sale_price_effective_date',
        'item_group_id',
        'gender',
        'color',
        'size',
        'age_group',
        'material',
        'pattern',
        'shipping',
        'shipping_weight',
        'style[0]',

    );

    /**
     * @inheritDoc
     */
    protected function configure()
    {
        $this->setName('catalog:facebookfeed');
        $this->setDescription('Generates a product feed for facebook');

        parent::configure();
    }

    /**
     * FacebookFeed constructor.
     * @param State $state
     * @param ProductRepository $productRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param DirectoryList $directoryList
     * @param Filesystem $filesystem
     * @param ImageFactory $imageHelperFactory
     * @param StockItemRepository $stockItemRepository
     * @param LinkManagementInterface $linkManagement
     */
    public function __construct(
        State $state,
        ProductRepository $productRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        DirectoryList $directoryList,
        Filesystem $filesystem,
        ImageFactory $imageHelperFactory,
        StockItemRepository $stockItemRepository,
        LinkManagementInterface $linkManagement,
        StoreManagerInterface $storeManager
    )
    {
        $this->state = $state;
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->directoryList = $directoryList;
        $this->filesystem = $filesystem;
        $this->imageHelperFactory = $imageHelperFactory;
        $this->stockItemRepository = $stockItemRepository;
        $this->linkManagement = $linkManagement;
        $this->storeManager = $storeManager;

        parent::__construct();
    }


    /**
     * Execute the command
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return null|int
     */
    protected function execute(
        InputInterface $input,
        OutputInterface $output
    )
    {
        $this->state->setAreaCode(\Magento\Framework\App\Area::AREA_ADMINHTML);
        $criteria = $this->searchCriteriaBuilder
            ->addFilter('status', '1', 'eq')
            ->addFilter('type_id','configurable','eq')
            ->addFilter('visibility', '4', 'eq')
            ->create();

        $products = $this->productRepository->getList($criteria)->getItems();

        $productRows['header'] = $this->generateRow($this->columns);

        foreach ($products as $key => $product) {
            $_product = $this->productRepository->getById($product->getId());
            $data = $this->generateProductData($_product);
            if (count($data) > 0) {
                $productRows[$key] = $this->generateRow($data);
            }
        }

        $feedContent = '';
        foreach ($productRows as $productRow) {
            $feedContent .= $productRow . "\n";
        }

        $media = $this->filesystem->getDirectoryWrite($this->directoryList::PUB);
        $media->writeFile("feed/facebook.csv", $feedContent);

        $output->writeln('<info>Facebook Feed generated in /pub/feed/facebook.csv</info>');
    }

    /**
     * @param $productId
     * @return mixed
     */
    public function getStockItem($productId)
    {
        return $this->stockItemRepository->get($productId);
    }

    /**
     * @param $rowData
     * @return string
     */
    public function generateRow($rowData)
    {
        $row = '';
        foreach ($rowData as $column) {
            $row .= '"' . $column . '",';
        }
        return substr($row, 0, -1);
    }

    /**
     * @param $product
     * @return array
     */
    public function generateProductData($product)
    {
        $data = array();
        if ($product) {
            $data['id'] = $product->getSku();
            $data['title'] = str_replace('"', '\'', $product->getName());
            $data['description'] = strip_tags(str_replace('"', '\'', $product->getShortDescription()));
            $data['availability'] = "In Stock";
            $data['condition'] = 'new';

            if ($product->getTypeId() == 'configurable') {
                $basePrice = $product->getPriceInfo()->getPrice('regular_price');
                $regular_price = $basePrice->getMinRegularAmount()->getValue();
                $finalPriceAmt = $product->getFinalPrice();
            } else {
                $regular_price = $product->getPriceInfo()->getPrice('regular_price')->getValue();
                $finalPriceAmt = $product->getPriceInfo()->getPrice('final_price')->getValue();
            }
            $data['price'] = number_format($regular_price,2) . ' USD';
            $currentStore = $this->storeManager->getStore();
            $mediaUrl = $currentStore->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
            $data['link'] = $product->getProductUrl();
            $data['image_link'] = $mediaUrl . 'catalog/product' . $product->getImage();
            $data['brand'] = "";
            $data['google_product_category'] = '';
            $data['fb_product_category'] = '';
            $data['quantity_to_sell_on_facebook'] = '';
            $data['sale_price'] = number_format($finalPriceAmt,2) . ' USD';
            $data['sale_price_effective_date'] = '';
            $data['item_group_id'] = '';
            $data['gender'] = '';
            $data['color'] = '';
            $data['size'] = '';
            $data['age_group'] = '';
            $data['material'] = '';
            $data['pattern'] = '';
            $data['shipping'] = '';
            $data['shipping_weight'] = '';
            $data['style[0]'] = '';
        }
        return $data;
    }

    /**
     * Get Store name
     *
     * @return string
     */
    public function getStoreName()
    {
        return $this->storeManager->getStore()->getName();
    }
}
