<?php

namespace Joshine\GoogleFeed\Console\Command;

use Magento\Framework\App\State;
use Magento\Store\Model\Store;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Catalog\Model\ProductRepository;
use Magento\Framework\Filesystem;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Catalog\Helper\Image as ImageFactory;
use Magento\CatalogInventory\Model\Stock\StockItemRepository;
use Magento\ConfigurableProduct\Api\LinkManagementInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\CategoryList;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Eav\Api\AttributeRepositoryInterface;
/**
 * Class SomeCommand
 */
class GoogleFeed extends Command
{
    /* @var ProductRepository */
    private $productRepository;
    /* var SearchCriteriaBuilder */
    private $searchCriteriaBuilder;
    /* var DirectoryList */
    private $directoryList;
    /* var Filesystem */
    private $filesystem;
    /* @var ImageFactory */
    protected $imageHelperFactory;
    /* @var StockItemRepository */
    protected $stockItemRepository;
    /* @var LinkManagementInterface */
    protected $linkManagement;
    /* @var StoreManagerInterface */
    protected $storeManager;
    /* @var CategoryList */
    private $categoryList;
    /* @var ScopeConfigInterface */
    private $scopeConfigInterface;
    /* @var AttributeRepositoryInterface */
    private $attributeRepositoryInterface;

    /*table title*/
    private $columns = array(
        'id',
        'title',
        'description',
        'google product category',
        'product type',
        'link',
        'mobile link',
        'image link',
        'additional image link',
        'condition',
        'availability',
        'price',
        'brand',
        'mpn',
        'gender',
        'age group',
        'color',
        'size system',
        'size',
        'item group id',
        'material',
        'shipping(country:price)',
        'shipping weight',
        'custom label 0',
        'custom label 1',
        'custom label 2',
        'custom label 3'
    );

    /*color*/
    private $color = [
        'RED' => 'Red',
        'YLW' => 'Yellow',
        'ORG' => 'Orange',
        'KAK' => 'Khaki',
        'NAV' => 'Navy',
        'SLV' => 'Silver',
        'LGY' => 'Light Grey',
        'LGN' => 'Light Green',
        'LCR' => 'Light Coral',
        'VRD' => 'Violet Red',
        'CBU' => 'Cadet Blue',
        'CYN' => 'Cyan',
        'CSK' => 'Cornsilk',
        'FSA' => 'Fuchsia',
        'GYW' => 'Green Yellow',
        'HPK' => 'Hot Pink',
        'IVY' => 'Ivory',
        'LCN' => 'Light Cyan',
        'LIN' => 'Linen',
        'IDG' => 'Indigo'
    ];

    private $pattern = [
        'one piece',
        'bikini',
        'cover up'
    ];
    /**
     * @inheritDoc
     */
    protected function configure()
    {
        $this->setName('catalog:googlefeed');
        $this->setDescription('Generates a product feed for google');

        parent::configure();
    }

    /**
     * GoogleFeed constructor.
     * @param State $state
     * @param ProductRepository $productRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param DirectoryList $directoryList
     * @param Filesystem $filesystem
     * @param ImageFactory $imageHelperFactory
     * @param StockItemRepository $stockItemRepository
     * @param LinkManagementInterface $linkManagement
     * @param CategoryList $categoryList
     * @param ScopeConfigInterface $scopeConfigInterface
     */
    public function __construct(
        State $state,
        ProductRepository $productRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        DirectoryList $directoryList,
        Filesystem $filesystem,
        ImageFactory $imageHelperFactory,
        StockItemRepository $stockItemRepository,
        LinkManagementInterface $linkManagement,
        StoreManagerInterface $storeManager,
        CategoryList $categoryList,
        ScopeConfigInterface $scopeConfigInterface,
        AttributeRepositoryInterface $attributeRepositoryInterface
    )
    {
        $this->state = $state;
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->directoryList = $directoryList;
        $this->filesystem = $filesystem;
        $this->imageHelperFactory = $imageHelperFactory;
        $this->stockItemRepository = $stockItemRepository;
        $this->linkManagement = $linkManagement;
        $this->storeManager = $storeManager;
        $this->categoryList = $categoryList;
        $this->scopeConfigInterface = $scopeConfigInterface;
        $this->attributeRepositoryInterface = $attributeRepositoryInterface;
        parent::__construct();
    }


    /**
     * Execute the command
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return null|int
     */
    protected function execute(
        InputInterface $input,
        OutputInterface $output
    )
    {
        $this->state->setAreaCode(\Magento\Framework\App\Area::AREA_ADMINHTML);
        $categoryCriteria = $this->searchCriteriaBuilder
            ->addFilter('is_create_google_feed', '1', 'eq')
            ->create();

        $category = $this->categoryList->getList($categoryCriteria)->getItems();
        $categoryId = [];
        $categoryGoogleProductCategory = [];
        foreach ($category as $key=>$category){
            $categoryId[] =  $category->getId();
            $categoryGoogleProductCategory[$category->getId()]['GoogleProductCategory'] = $category->getGoogleProductCategory();
            $categoryGoogleProductCategory[$category->getId()]['name'] = $category->getName();
            $name = $category->getName().'-';
            $name = $this->getPrdouctType($category,$name);
            $name = implode('->',array_reverse(explode('-',rtrim($name,'-'))));
            $categoryGoogleProductCategory[$category->getId()]['product type'] = ltrim($name,'Category->') ;
        }


        $criteria = $this->searchCriteriaBuilder
            ->addFilter('status', '1', 'eq')
            ->addFilter('visibility', '4', 'eq')
            ->addFilter('type_id','simple','eq')
            ->addFilter('category_id',$categoryId,'in')
            ->create();

        $products = $this->productRepository->getList($criteria)->getItems();

        $productRows['header'] = $this->generateRow($this->columns);

        foreach ($products as $key => $product) {
            $_product = $this->productRepository->getById($product->getId());
            $data = $this->generateProductData($_product,$categoryGoogleProductCategory);
            if (count($data) > 0) {
                $productRows[$key] = $this->generateRow($data);
            }
        }

        $feedContent = '';
        foreach ($productRows as $productRow) {
            $feedContent .= $productRow . "\n";
        }

        $media = $this->filesystem->getDirectoryWrite($this->directoryList::PUB);
        $media->writeFile("feed/google.csv", $feedContent);

        $output->writeln('<info>Google Feed generated in /pub/feed/google.csv</info>');
    }

    /**
     * @param $productId
     * @return mixed
     */
    public function getStockItem($productId)
    {
        return $this->stockItemRepository->get($productId);
    }

    /**
     * @param $rowData
     * @return string
     */
    public function generateRow($rowData)
    {
        $row = '';
        foreach ($rowData as $column) {
            $row .= '"' . $column . '",';
        }
        return substr($row, 0, -1);
    }

    /**
     * @param $product
     * @return array
     */
    public function generateProductData($product,$categoryGoogleProductCategory)
    {
        $data = array();
        if ($product) {
            $data['id'] = $product->getSku();
            $data['title'] = $this->getStoreName() .' '. str_replace('"', '\'', $product->getName());
            $data['description'] = strip_tags(str_replace('"', '\'', $product->getDescription()));


            foreach( $product->getCategoryIds() as $k=>$v){
                if (isset($categoryGoogleProductCategory[$v])){
                    $categoryId = $v;
                    break;
                }
            }

            $data['google product category'] = $categoryGoogleProductCategory[$categoryId]['GoogleProductCategory'];
            $data['product type'] = $categoryGoogleProductCategory[$categoryId]['product type'];
            $data['link'] = $product->getProductUrl();
            $data['mobile link'] = $product->getProductUrl();

            $currentStore = $this->storeManager->getStore();
            $mediaUrl = $currentStore->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);

            $data['image link'] = $mediaUrl . 'catalog/product' . $product->getImage();
            $data['additional image link'] = '';
            $data['condition'] = 'new';
            $data['availability'] = 'in stock';

            if ($product->getTypeId() == 'configurable') {
                $basePrice = $product->getPriceInfo()->getPrice('regular_price');
                $regular_price = $basePrice->getMinRegularAmount()->getValue();
                $finalPriceAmt = $product->getFinalPrice();
            } else {
                $regular_price = $product->getPriceInfo()->getPrice('regular_price')->getValue();
                $finalPriceAmt = $product->getPriceInfo()->getPrice('final_price')->getValue();
            }

            $data['price'] = number_format($finalPriceAmt,2) . ' USD';
            $data['brand'] = $this->getStoreName();
            $mpu = explode('-',$product->getSku())[0];
            $data['mpn'] = $mpu;
            $data['gender'] = 'Female';
            $data['age group'] = 'Adult';

            $attribute = $this->attributeRepositoryInterface->get('4', 'color');
            $optionText = $attribute->getSource()->getOptionText($product->getColor());

            if (in_array($optionText,$this->pattern)){
                $colorShort = $this->explode('-',$product->getSku())[1];
                $data['color'] = $this->color[$colorShort];
            }else{
                $data['color'] = $optionText;
            }

            $data['size system'] = '';
            $attribute = $this->attributeRepositoryInterface->get('4', 'size');
            $sizeText = $attribute->getSource()->getOptionText($product->getSize());
            $data['size'] = $sizeText;
            $data['item group id'] = $mpu;
            $data['material'] = '';
            $data['shipping(country:price)'] =':'. $this->scopeConfigInterface->getValue('carriers/flatrate/price') . ' USD';
            $data['shipping weight'] = $product->getWeight();
            $data['custom label 0'] = '';
            $data['custom label 1'] = '';
            $data['custom label 2'] = '';
            $data['custom label 3'] = '';

        }
        return $data;
    }

    /**
     * Get Store name
     *
     * @return string
     */
    public function getStoreName()
    {
        return $this->storeManager->getStore()->getName();
    }

    public function getPrdouctType($category,&$name){
        $parent = $category->getParentCategory();
        $name .= $parent->getName().'-';
        if ($parent->getParentCategory()->getName() != 'Root Catalog'){
            $this->getPrdouctType($parent,$name);
        }

        return $name;
    }
}
