<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SalesRule\Model\Rule\Action\Discount;

use Magento\SalesRule\Model\Rule\Action\Discount\AbstractDiscount;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\SalesRule\Model\DeltaPriceRound;
use Magento\SalesRule\Model\Rule\Action\Discount\DataFactory;
use Magento\SalesRule\Model\Validator;

class BuyXGetY extends AbstractDiscount
{
    /**
     * @var DeltaPriceRound
     */
    private $deltaPriceRound;
    private static $discountType = 'BuyXGetY';

    /**
     * @param Validator $validator
     * @param DataFactory $discountDataFactory
     * @param PriceCurrencyInterface $priceCurrency
     * @param DeltaPriceRound $deltaPriceRound
     */
    public function __construct(
        Validator $validator,
        DataFactory $discountDataFactory,
        PriceCurrencyInterface $priceCurrency,
        DeltaPriceRound $deltaPriceRound = null
    ) {
        $this->deltaPriceRound = $deltaPriceRound ?: ObjectManager::getInstance()->get(DeltaPriceRound::class);
        parent::__construct($validator, $discountDataFactory, $priceCurrency);
    }

    /**
     * @param \Magento\SalesRule\Model\Rule $rule
     * @param \Magento\Quote\Model\Quote\Item\AbstractItem $item
     * @param float $qty
     * @return \Magento\SalesRule\Model\Rule\Action\Discount\Data
     */
    public function calculate($rule, $item, $qty)
    {
        /** @var \Magento\SalesRule\Model\Rule\Action\Discount\Data $discountData */
        $discountData = $this->discountFactory->create();

        $ruleId = $rule->getId();
        $ruleTotals = $this->validator->getRuleItemTotalsInfo($ruleId);
        $quote = $item->getQuote();
        $address = $item->getAddress();

        $itemPrice = $this->validator->getItemPrice($item);
        $baseItemPrice = $this->validator->getItemBasePrice($item);
        $itemOriginalPrice = $this->validator->getItemOriginalPrice($item);
        $baseItemOriginalPrice = $this->validator->getItemBaseOriginalPrice($item);

        $cartRules = $address->getBuyXGetYRules();
        if (!isset($cartRules[$ruleId])) {
            return $discountData;
        }
        $availableDiscountAmount = (float)$cartRules[$ruleId]['amount'];
        $ruldDiscountAmount = (float)$cartRules[$ruleId]['discount_amount'];
        $discountType = self::$discountType . $ruleId;

        if ($availableDiscountAmount > 0) {
            $store = $quote->getStore();
            if ($ruleTotals['items_count'] <= 1) {
                $quoteAmount = $this->priceCurrency->convert($availableDiscountAmount, $store);
                $baseDiscountAmount = min($baseItemPrice * $qty, $availableDiscountAmount);
                $this->deltaPriceRound->reset($discountType);
            } else {
                $ratio = $baseItemPrice * $qty / $ruleTotals['base_items_price'];
                $maximumItemDiscount = $this->deltaPriceRound->round(
                    $ruldDiscountAmount * $ratio,
                    $discountType
                );

                $quoteAmount = $this->priceCurrency->convert($maximumItemDiscount, $store);

                $baseDiscountAmount = min($baseItemPrice * $qty, $maximumItemDiscount);
                $this->validator->decrementRuleItemTotalsCount($ruleId);
            }

            $baseDiscountAmount = $this->priceCurrency->round($baseDiscountAmount);

            $availableDiscountAmount -= $baseDiscountAmount;
            $cartRules[$ruleId]['amount'] = $availableDiscountAmount;
            if ($availableDiscountAmount <= 0) {
                $this->deltaPriceRound->reset($discountType);
            }

            $discountData->setAmount($this->priceCurrency->round(min($itemPrice * $qty, $quoteAmount)));
            $discountData->setBaseAmount($baseDiscountAmount);
            $discountData->setOriginalAmount(min($itemOriginalPrice * $qty, $quoteAmount));
            $discountData->setBaseOriginalAmount($this->priceCurrency->round($baseItemOriginalPrice));
        }
        $address->setBuyXGetYRules($cartRules);

        return $discountData;
    }
}
