<?php

namespace Joshine\Review\Model\Repository;

use Joshine\Review\Block\Images;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Joshine\Review\Model\ResourceModel;
use Joshine\Review\Model\ImagesFactory;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Filesystem\DirectoryList;
use Joshine\Review\Helper\ImageHelper;

class ImagesRepository
{
    /**
     * @var array
     */
    private $images = [];

    /**
     * @var ResourceModel\Images
     */
    private $imagesResource;

    /**
     * @var ImagesFactory
     */
    private $imagesFactory;


    /**
     * @var ResourceModel\Images\CollectionFactory
     */
    private $imagesCollectionFactory;


    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $ioFile;

    /**
     * @var null|string
     */
    private $folderPath = null;

    /**
     * @var \Magento\Framework\Filesystem
     */
    private $filesystem;


    public function __construct(
        \Joshine\Review\Model\ResourceModel\Images $imagesResource,
        \Joshine\Review\Model\ImagesFactory $imagesFactory,
        \Joshine\Review\Model\ResourceModel\Images\CollectionFactory $imagesCollectionFactory,
        \Magento\Framework\Filesystem $filesystem,
        \Magento\Framework\Filesystem\Io\File $ioFile
    ) {
        $this->imagesResource = $imagesResource;
        $this->imagesFactory = $imagesFactory;
        $this->imagesCollectionFactory = $imagesCollectionFactory;
        $this->ioFile = $ioFile;
        $this->filesystem = $filesystem;
    }

    /**
     * {@inheritdoc}
     */
    public function get($imageId)
    {
        if (!isset($this->images[$imageId])) {
            /** @var \Joshine\Review\Model\Images $image */
            $image = $this->imagesFactory->create();
            $this->imagesResource->load($image, $imageId);
            if (!$image->getImageId()) {
                throw new NoSuchEntityException(__('Rule with specified ID "%1" not found.', $imageId));
            }
            $this->images[$imageId] = $image;
        }
        return $this->images[$imageId];
    }

    /**
     * @return string
     */
    private function getFolderPath()
    {
        if ($this->folderPath === null) {
            $this->folderPath = $this->filesystem->getDirectoryRead(
                DirectoryList::MEDIA
            )->getAbsolutePath(
                ImageHelper::IMAGE_PATH
            );
        }

        return $this->folderPath;
    }


    /**
     * {@inheritdoc}
     */
    public function delete( $image)
    {
        try {
            $this->ioFile->rm($this->getFolderPath() . $image->getPath());
            $this->ioFile->rm(
                $this->getFolderPath() . 'resized/' .  Images::REVIEW_COVER_WIDTH
            );
            $this->imagesResource->delete($image);
            unset($this->images[$image->getId()]);
        } catch (\Exception $e) {
            if ($image->getImageId()) {
                throw new CouldNotDeleteException(
                    __('Unable to remove image with ID %1. Error: %2', [$image->getImageId(), $e->getMessage()])
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove image. Error: %1', $e->getMessage()));
        }
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function deleteById($imageId)
    {
        $model = $this->get($imageId);
        $this->delete($model);
        return true;
    }

}
