<?php

namespace Joshine\Script\Console\Command;


use Joshine\Review\Helper\ImageHelper;
use Joshine\Review\Model\ImageUploader;
use Magento\Framework\App\Area;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\App\State;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\File\Csv;
use Magento\Framework\ObjectManager\ObjectManager;
use Magento\Setup\Exception;
use Magento\Store\Model\App\Emulation;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

use Magento\Review\Model\Review;

use Magento\Customer\Model\Customer;
use Magento\Customer\Model\ResourceModel\Customer\CollectionFactory;


class ReviewsImporter extends Command
{
    /**
     * @var Csv
     */
    private $csv;
    /**
     * @var Customer
     */
    private $customer;
    /**
     * @var DirectoryList
     */
    private $directoryList;
    /**
     * @var ResourceConnection
     */
    private $resource;
    /**
     * @var AdapterInterface
     */
    private $connection;
    /**
     * @var State
     */
    private $state;
    /**
     * @var Magento\Customer\Model\ResourceModel\Customer\Collection
     */
    private $collection;
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    const   FILE = "csv";

    /**
     * @var ObjectManager
     */
    private $objectManager;
    /**
     * @var OutputInterface
     */
    private $_output;
    /**
     * @var mixed
     */
    private $storeId;
    /**
     * @var ImageUploader
     */
    private $imageUploader;
    /**
     * @var \Joshine\Review\Model\ImagesFactory
     */
    private $imagesFactory;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var \Joshine\Review\Model\ResourceModel\Images
     */
    private $imagesResource;
    /**
     * @var \Magento\Framework\Filesystem
     */
    private $filesystem;


    public function __construct(
        \Magento\Framework\Filesystem $filesystem,
        ObjectManager      $objectManager,
        CollectionFactory  $collectionFactory,
        Csv                $csv,
        Customer           $customer,
        DirectoryList      $directoryList,
        ResourceConnection $resource,
        State              $state,
        \Magento\Framework\Filesystem\Io\File $ioFile,
        \Magento\Framework\App\RequestInterface $request,
        ImageUploader $imageUploader,
        \Joshine\Review\Model\ImagesFactory $imagesFactory,
        \Magento\Framework\Message\ManagerInterface $messageManager,
        \Joshine\Review\Model\ResourceModel\Images $imagesResource,
        string             $name = null
    )
    {
        parent::__construct($name);
        $this->imageUploader     = $imageUploader;
        $this->filesystem        = $filesystem;
        $this->imagesFactory     = $imagesFactory;
        $this->messageManager    = $messageManager;
        $this->imagesResource    = $imagesResource;
        $this->collectionFactory = $collectionFactory;
        $this->csv               = $csv;
        $this->customer          = $customer;
        $this->directoryList     = $directoryList;
        $this->state             = $state;
        $this->ioFile            = $ioFile;
    }

    protected function configure()
    {
        $this->setName('joshine:review-import');
        $this->setDescription('导入产品评论');
        $this->setDefinition([
            new InputOption('csv-path', null, InputOption::VALUE_OPTIONAL, 'import the csv file'),
        ]);
        $this->addOption(
            self::FILE,
            null,
            InputOption::VALUE_REQUIRED,
            'csv路径'
        );

        $this->addOption(
            "store_id",
            null,
            InputOption::VALUE_REQUIRED,
            'store id'
        );

        parent::configure();

    }

    protected function execute(
        InputInterface  $input,
        OutputInterface $output
    )
    {
        $this->_output = $output;
        try {
            $path = $input->getOption(self::FILE);
            if (!$path) {
                $output->writeln($this->errorWrapper("路径不对"));
                return;
            }

            $store_id = $input->getOption('store_id');
            if (!$store_id) {
                $output->writeln($this->errorWrapper("store_id没填"));
                return;
            }
            $this->storeId = $store_id;
            $this->importCsvFile($path);
        } catch (\Exception $e) {
            $output->writeln($this->errorWrapper($e->getMessage()));
        }
    }

    private function importCsvFile($filename)
    {
        $data = $this->loadAndCheck($filename);

        $data = $this->formatImagePath($data);

        foreach ($data as $row) {
            $this->insertDB($row);
        }

    }

    private function insertDB($data)
    {

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $reviewFactory = $objectManager->get("Magento\Review\Model\ReviewFactory");
        $ratingFactory = $objectManager->get("Magento\Review\Model\RatingFactory");
        $productId = $objectManager->get('Magento\Catalog\Model\Product')->getIdBySku(trim($data['sku']));
        if ($productId) {
            $review = $reviewFactory->create()->setData($data);
            $review->unsetData('review_id');
            $review->setEntityId($review->getEntityIdByCode(Review::ENTITY_PRODUCT_CODE))
                ->setEntityPkValue($productId)
                ->setStatusId(Review::STATUS_PENDING)
                ->setStoreId($this->storeId)
                ->setStores([$this->storeId])
                ->save();
            $this->storeImage($data['image'], $review->getId());
            $rating = [];
            //前端实测
            $rating[4] = strval($data['ratings'] + 15);
            foreach ($rating as $ratingId => $optionId) {
                $ratingFactory->create()
                    ->setRatingId($ratingId)
                    ->setReviewId($review->getId())
                    ->addOptionVote($optionId, $productId);
            }
            $review->aggregate();
        } else {
            $this->_output->writeln($this->errorWrapper("sku : {$data['sku']} 产品未找到"));
        }

    }

    private function storeImage($images, $id)
    {
        foreach ($images as $image) {
            $result = $this->copy($image);
            if ($result) {
                $this->saveImage($result, $id);
            }
        }
    }

    public function copy(string $imagePath)
    {
        $from = $imagePath;
        $fileName = basename($imagePath);
        if ($this->ioFile->fileExists($from)) {
            $realPath = $this->filesystem->getDirectoryRead(
                DirectoryList::MEDIA
            )->getAbsolutePath(
                ImageHelper::IMAGE_PATH
            );
            $counter = 0;
            while ($this->ioFile->fileExists($realPath . $fileName)) {
                $imagePathArray = explode('.', $fileName);
                $imagePathArray[0] .= $counter++;
                $fileName = implode('.', $imagePathArray);
            }
            $this->ioFile->checkAndCreateFolder($this->ioFile->dirname($realPath . $fileName));
            if ($this->ioFile->mv($from, $realPath . $fileName)) {
                return  '/'.basename($fileName);
            }
        }

        return false;
    }

    private function formatImagePath($data)
    {
        foreach ($data as $index => $row) {
            if (!empty($row["image"])) {
                $urls = array_filter(explode("https://", $row["image"]));
                $path = $this->downLoadImage($urls);
                $data[$index]['image'] = $path;
            }
        }
        return $data;
    }

    private function downLoadImage($urls)
    {
        $res = [];
        $path = $this->directoryList->getPath('var');
        foreach ($urls as $index => $url) {
            $url = trim($url, ",");
            $url = trim($url, " ");
            $url = trim($url, ",");
            $url = trim($url, ",");
            $url = trim($url, ".");
            $url = trim($url, "．");
            $url = trim($url, "，");
            $file = $path.DIRECTORY_SEPARATOR.mt_rand(10000,99999).time().md5($url).".jpeg";
            if(!file_exists($file)){
                $this->_output->writeln($this->successWrapper("图片${url}下载中..."));
                try {
                    file_put_contents($file,file_get_contents("https://".$url));
                    $res[] = $file;
                } catch (\Exception $e) {
                    $output->writeln($this->errorWrapper($e->getMessage()));
                }
            }else{
                $this->_output->writeln($this->errorWrapper("图片${url}已下载过"));
                $res[] = $file;
            }
        }
        return $res;
    }

    private function loadAndCheck($filename)
    {
        $file = $this->directoryList->getPath('var') . DIRECTORY_SEPARATOR . $filename;
        $csvData = fopen($file, 'r');
        $data = [];
        $firstLine = true;
        while ($row = fgetcsv($csvData)) {
            if ($firstLine == true) {
                $this->checkHeader($row);
                $firstLine = false;
                continue;
            }
            $review['sku']        = trim($row[0]);
            $review['nickname']   = trim($row[1]);
            $review['ratings']    = trim($row[2]);
            $review['title']      = trim($row[3]);
            $review['detail']     = trim($row[4]);
            $review['image']      = trim($row[5]);
            $review['size_fits']  = trim($row[6]);
            $review['date']       = trim($row[7]);
            if (empty(trim($review["sku"]))
                || empty(trim($review["detail"]))
                || empty(trim($review['nickname']))
                || empty($review['date'])
                || empty($review['title']
                || empty($review['size_fits'])
                || empty($review['ratings'])
                )
            ) {
                throw new \Exception("表格数据有空!");
            }
            $data[] = $review;
        }
        fclose($csvData);
        return $data;
    }

    private function successWrapper($msg)
    {
        return "<info>$msg</info>";
    }

    private function errorWrapper($msg)
    {
        return "<error>$msg</error>";
    }

    private function checkHeader($header)
    {
        $field = [
            'size_fits',
            'sku',
            'name',
            'content',
            'rating',
            'image',
            'date',
            'title'
        ];
        foreach ($field as $row) {
            if (!in_array($row, $header)) {
                throw new \Exception("表格字段不全！,必须包含".implode(' ',$field));
            }
        }
    }


    /**
     * @return array
     */
    private function getHeaders($line)
    {
        if (count($line) <  8 ) {
            throw new \Exception("csv格式不对！");
        }

        if ($line[0] != 'sku') {
            throw new \Exception("第1列应该是sku！");
        }

        if ($line[1] != 'name') {
            throw new \Exception('第2列应该是name');
        }

        if ($line[2] != 'rating') {
            throw new \Exception('第3列应该是rating');
        }

        if ($line[3] != 'title') {
            throw new \Exception('第4列应该是rating');
        }

        if ($line[4] != 'content') {
            throw new \Exception('第5列应该是content');
        }

        if ($line[5] != 'image') {
            throw new \Exception('第6列应该是image');
        }

        if ($line[6] != 'size_fits') {
            throw new \Exception('第7列应该是size_fits');
        }

        if ($line[7] != 'date') {
            throw new \Exception('第8列应该是date');
        }
        return $line;
    }



    private function saveImage($result, $reviewId)
    {
        $model = $this->imagesFactory->create();
        $model->setReviewId($reviewId);
        $model->setPath($result);

        if ($model->getImageId()) {
            $model = $this->getFullImage($model->getImageId())->addData($model->getData());
        }
        $this->imagesResource->save($model);
    }

    private function getFullImage($imageId) {
        $image = $this->imagesFactory->create();
        $this->imagesResource->load($image, $imageId);
        if (!$image->getImageId()) {
            throw new NoSuchEntityException(__('Rule with specified ID "%1" not found.', $imageId));
        }
        return $image;
    }
}
