import { Ref, onMounted, ref } from 'vue'

import type { PaginationData } from '@/types/api'

export interface UsePageListOptions<T> {
  query: (
    currentPage: number,
    pageSize: number,
    statusCode?: number,
  ) => Promise<PaginationData<T>>
  initPageSize?: number
  statistics?: boolean
  initLoad?: boolean
}

export default function usePageList<T>(options: UsePageListOptions<T>) {
  const loading = ref(true)
  const currentPage = ref(1)
  const pageSize = ref(options.initPageSize || 100)
  const total = ref(0)
  const data = ref<T[]>([]) as Ref<T[]>
  const statistics = ref(options.statistics || false)
  const totalData = ref({
    sumShipmentNum: 0,
    sumPassNum: 0,
    sumTotalPrice: 0,
    sumNotPassNum: 0,
  })
  const loadData = async () => {
    const { query } = options
    try {
      loading.value = true
      // Promise 链式调用
      const res = await query(currentPage.value, pageSize.value)
      if (statistics.value) {
        totalData.value = res as never
        total.value = res.page!.total
        currentPage.value = res.page!.current
        pageSize.value = res.page!.size
        data.value = res.page!.records
      } else {
        total.value = res.total
        currentPage.value = res.current
        pageSize.value = res.size
        data.value = res.records
      }
    } catch (error) {
      console.error(error)
      // showError(error)
    } finally {
      loading.value = false
    }
  }

  const onCurrentPageChange = (page: number) => {
    currentPage.value = page
    loadData()
  }

  const refresh = (reset?: boolean) => {
    if (reset) {
      data.value = []
      total.value = 0
    }
    currentPage.value = 1
    loadData()
  }

  const onPageSizeChange = (newPageSize: number) => {
    pageSize.value = newPageSize
    refresh()
  }

  onMounted(() => {
    if (options.initLoad !== false) {
      refresh()
    }
  })

  return {
    loading,
    currentPage,
    totalData,
    pageSize,
    total,
    data,
    onCurrentPageChange,
    onPageSizeChange,
    refresh,
  }
}
